/*
 * LibrePCB - Professional EDA for everyone!
 * Copyright (C) 2013 LibrePCB Developers, see AUTHORS.md for contributors.
 * https://librepcb.org/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef LIBREPCB_CORE_NETLINE_H
#define LIBREPCB_CORE_NETLINE_H

/*******************************************************************************
 *  Includes
 ******************************************************************************/
#include "../serialization/serializableobjectlist.h"
#include "../types/length.h"

#include <QtCore>

/*******************************************************************************
 *  Namespace / Forward Declarations
 ******************************************************************************/
namespace librepcb {

/*******************************************************************************
 *  Class NetLineAnchor
 ******************************************************************************/

/**
 * @brief The NetLineAnchor class
 */
class NetLineAnchor final {
  Q_DECLARE_TR_FUNCTIONS(NetLineAnchor)

public:
  // Types
  struct BusAnchor {
    Uuid segment;
    Uuid junction;

    bool operator==(const BusAnchor& rhs) const noexcept {
      return (segment == rhs.segment) && (junction == rhs.junction);
    }
  };
  struct PinAnchor {
    Uuid symbol;
    Uuid pin;

    bool operator==(const PinAnchor& rhs) const noexcept {
      return (symbol == rhs.symbol) && (pin == rhs.pin);
    }
  };

  // Constructors / Destructor
  NetLineAnchor() = delete;
  NetLineAnchor(const NetLineAnchor& other) noexcept;
  explicit NetLineAnchor(const SExpression& node);
  ~NetLineAnchor() noexcept;

  // Getters
  const std::optional<Uuid>& tryGetJunction() const noexcept {
    return mJunction;
  }
  const std::optional<BusAnchor>& tryGetBusJunction() const noexcept {
    return mBusJunction;
  }
  const std::optional<PinAnchor>& tryGetPin() const noexcept { return mPin; }

  // General Methods

  /**
   * @brief Serialize into ::librepcb::SExpression node
   *
   * @param root    Root node to serialize into.
   */
  void serialize(SExpression& root) const;

  // Operator Overloadings
  bool operator==(const NetLineAnchor& rhs) const noexcept;
  bool operator!=(const NetLineAnchor& rhs) const noexcept {
    return !(*this == rhs);
  }
  bool operator<(const NetLineAnchor& rhs) const noexcept;
  NetLineAnchor& operator=(const NetLineAnchor& rhs) noexcept;

  // Static Methods
  static NetLineAnchor junction(const Uuid& junction) noexcept;
  static NetLineAnchor busJunction(const Uuid& segment,
                                   const Uuid& junction) noexcept;
  static NetLineAnchor pin(const Uuid& symbol, const Uuid& pin) noexcept;

private:  // Methods
  NetLineAnchor(const std::optional<Uuid>& junction,
                const std::optional<BusAnchor> bus,
                const std::optional<PinAnchor>& pin) noexcept;

private:  // Data
  std::optional<Uuid> mJunction;
  std::optional<BusAnchor> mBusJunction;
  std::optional<PinAnchor> mPin;
};

/*******************************************************************************
 *  Class NetLine
 ******************************************************************************/

/**
 * @brief The NetLine class represents a net line within a schematic
 *
 * The main purpose of this class is to serialize and deserialize schematic
 * net lines.
 *
 * @note The order of anchors (P1 & P2) is deterministic (sorted) to ensure a
 *       canonical file format & behavior. The constructor and #setAnchors()
 *       will automatically swap the passed anchors if needed.
 */
class NetLine final {
  Q_DECLARE_TR_FUNCTIONS(NetLine)

public:
  // Signals
  enum class Event {
    UuidChanged,
    WidthChanged,
    AnchorsChanged,
  };
  Signal<NetLine, Event> onEdited;
  typedef Slot<NetLine, Event> OnEditedSlot;

  // Constructors / Destructor
  NetLine() = delete;
  NetLine(const NetLine& other) noexcept;
  NetLine(const Uuid& uuid, const NetLine& other) noexcept;
  NetLine(const Uuid& uuid, const UnsignedLength& width, const NetLineAnchor& a,
          const NetLineAnchor& b) noexcept;
  explicit NetLine(const SExpression& node);
  ~NetLine() noexcept;

  // Getters
  const Uuid& getUuid() const noexcept { return mUuid; }
  const UnsignedLength& getWidth() const noexcept { return mWidth; }
  const NetLineAnchor& getP1() const noexcept { return mP1; }
  const NetLineAnchor& getP2() const noexcept { return mP2; }

  // Setters
  bool setUuid(const Uuid& uuid) noexcept;
  bool setWidth(const UnsignedLength& width) noexcept;
  bool setAnchors(NetLineAnchor a, NetLineAnchor b) noexcept;

  // General Methods

  /**
   * @brief Serialize into ::librepcb::SExpression node
   *
   * @param root    Root node to serialize into.
   */
  void serialize(SExpression& root) const;

  // Operator Overloadings
  bool operator==(const NetLine& rhs) const noexcept;
  bool operator!=(const NetLine& rhs) const noexcept { return !(*this == rhs); }
  NetLine& operator=(const NetLine& rhs) noexcept;

private:  // Methods
  static void normalizeAnchors(NetLineAnchor& start,
                               NetLineAnchor& end) noexcept;

private:  // Data
  Uuid mUuid;
  UnsignedLength mWidth;
  NetLineAnchor mP1;
  NetLineAnchor mP2;
};

/*******************************************************************************
 *  Class NetLineList
 ******************************************************************************/

struct NetLineListNameProvider {
  static constexpr const char* tagname = "line";
};
using NetLineList =
    SerializableObjectList<NetLine, NetLineListNameProvider, NetLine::Event>;

/*******************************************************************************
 *  Non-Member Functions
 ******************************************************************************/

inline std::size_t qHash(const NetLineAnchor& key,
                         std::size_t seed = 0) noexcept {
  QString s;
  if (std::optional<Uuid> anchor = key.tryGetJunction()) {
    s += anchor->toStr();
  }
  if (std::optional<NetLineAnchor::BusAnchor> anchor =
          key.tryGetBusJunction()) {
    s += anchor->segment.toStr();
    s += anchor->junction.toStr();
  }
  if (std::optional<NetLineAnchor::PinAnchor> anchor = key.tryGetPin()) {
    s += anchor->symbol.toStr();
    s += anchor->pin.toStr();
  }
  Q_ASSERT(!s.isEmpty());

  return ::qHash(s, seed);
}

/*******************************************************************************
 *  End of File
 ******************************************************************************/

}  // namespace librepcb

#endif
