/*
 * LibrePCB - Professional EDA for everyone!
 * Copyright (C) 2013 LibrePCB Developers, see AUTHORS.md for contributors.
 * https://librepcb.org/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef LIBREPCB_EDITOR_SYMBOLGRAPHICSITEM_H
#define LIBREPCB_EDITOR_SYMBOLGRAPHICSITEM_H

/*******************************************************************************
 *  Includes
 ******************************************************************************/
#include <librepcb/core/library/cmp/component.h>
#include <librepcb/core/library/cmp/componentsymbolvariantitem.h>
#include <librepcb/core/library/sym/symbol.h>

#include <QtCore>
#include <QtWidgets>

#include <memory>

/*******************************************************************************
 *  Namespace / Forward Declarations
 ******************************************************************************/
namespace librepcb {
namespace editor {

class CircleGraphicsItem;
class GraphicsLayerList;
class ImageGraphicsItem;
class PolygonGraphicsItem;
class SymbolPinGraphicsItem;
class TextGraphicsItem;

/*******************************************************************************
 *  Class SymbolGraphicsItem
 ******************************************************************************/

/**
 * @brief The SymbolGraphicsItem class
 */
class SymbolGraphicsItem final : public QGraphicsItemGroup {
public:
  enum class FindFlag {
    // Item types
    Pins = (1 << 0),
    Circles = (1 << 1),
    Polygons = (1 << 2),
    Texts = (1 << 3),
    Images = (1 << 4),
    All = Pins | Circles | Polygons | Texts | Images,

    // Match behavior
    AcceptNearMatch = (1 << 10),
  };
  Q_DECLARE_FLAGS(FindFlags, FindFlag)

  // Constructors / Destructor
  SymbolGraphicsItem() = delete;
  SymbolGraphicsItem(const SymbolGraphicsItem& other) = delete;
  SymbolGraphicsItem(Symbol& symbol, const GraphicsLayerList& layers,
                     QPointer<const Component> cmp,
                     std::shared_ptr<const ComponentSymbolVariantItem> cmpItem,
                     const QStringList& localeOrder,
                     bool hideUnusedPins) noexcept;
  ~SymbolGraphicsItem() noexcept;

  // Getters
  std::shared_ptr<SymbolPinGraphicsItem> getGraphicsItem(
      std::shared_ptr<SymbolPin> pin) noexcept {
    return mPinGraphicsItems.value(pin);
  }
  std::shared_ptr<CircleGraphicsItem> getGraphicsItem(
      std::shared_ptr<Circle> circle) noexcept {
    return mCircleGraphicsItems.value(circle);
  }
  std::shared_ptr<PolygonGraphicsItem> getGraphicsItem(
      std::shared_ptr<Polygon> polygon) noexcept {
    return mPolygonGraphicsItems.value(polygon);
  }
  std::shared_ptr<TextGraphicsItem> getGraphicsItem(
      std::shared_ptr<Text> text) noexcept {
    return mTextGraphicsItems.value(text);
  }
  std::shared_ptr<ImageGraphicsItem> getGraphicsItem(
      std::shared_ptr<Image> iamge) noexcept {
    return mImageGraphicsItems.value(iamge);
  }
  QList<std::shared_ptr<SymbolPinGraphicsItem>> getSelectedPins() noexcept;
  QList<std::shared_ptr<CircleGraphicsItem>> getSelectedCircles() noexcept;
  QList<std::shared_ptr<PolygonGraphicsItem>> getSelectedPolygons() noexcept;
  QList<std::shared_ptr<TextGraphicsItem>> getSelectedTexts() noexcept;
  QList<std::shared_ptr<ImageGraphicsItem>> getSelectedImages() noexcept;
  QList<std::shared_ptr<QGraphicsItem>> findItemsAtPos(
      const QPainterPath& posAreaSmall, const QPainterPath& posAreaLarge,
      FindFlags flags) noexcept;

  // Setters
  void setPosition(const Point& pos) noexcept;
  void setRotation(const Angle& rot) noexcept;

  // General Methods
  void updateAllTexts() noexcept;
  void setSelectionRect(const QRectF rect) noexcept;

  // Operator Overloadings
  SymbolGraphicsItem& operator=(const SymbolGraphicsItem& rhs) = delete;

private:  // Methods
  void syncPins() noexcept;
  void syncCircles() noexcept;
  void syncPolygons() noexcept;
  void syncTexts() noexcept;
  void syncImages() noexcept;
  void symbolEdited(const Symbol& symbol, Symbol::Event event) noexcept;
  void substituteText(TextGraphicsItem& text) noexcept;

private:  // Data
  Symbol& mSymbol;
  const GraphicsLayerList& mLayers;
  QPointer<const Component> mComponent;  // Can be nullptr.
  std::shared_ptr<const ComponentSymbolVariantItem> mItem;  // Can be nullptr.
  const QStringList mLocaleOrder;
  const bool mHideUnusedPins;
  QMap<std::shared_ptr<SymbolPin>, std::shared_ptr<SymbolPinGraphicsItem>>
      mPinGraphicsItems;
  QMap<std::shared_ptr<Circle>, std::shared_ptr<CircleGraphicsItem>>
      mCircleGraphicsItems;
  QMap<std::shared_ptr<Polygon>, std::shared_ptr<PolygonGraphicsItem>>
      mPolygonGraphicsItems;
  QMap<std::shared_ptr<Text>, std::shared_ptr<TextGraphicsItem>>
      mTextGraphicsItems;
  QMap<std::shared_ptr<Image>, std::shared_ptr<ImageGraphicsItem>>
      mImageGraphicsItems;

  // Slots
  Symbol::OnEditedSlot mOnEditedSlot;
};

}  // namespace editor
}  // namespace librepcb

Q_DECLARE_OPERATORS_FOR_FLAGS(librepcb::editor::SymbolGraphicsItem::FindFlags)

/*******************************************************************************
 *  End of File
 ******************************************************************************/

#endif
